/* --------------------------------------------------------------
  TextBox.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

/**
 * TextBox Widget
 *
 * This widget will handle the plain text style settings.
 */
export default class TextBox {
	/**
	 * Class Constructor 
	 * 
	 * @param {Setting} entity Contains the setting information.
	 * @param {StyleConfiguration} styleConfiguration Reference to the current configuration object.
	 */
	constructor(entity, styleConfiguration) {
		StyleEdit.Validator.isObject(entity);
		StyleEdit.Validator.isObject(styleConfiguration);
		
		/**
		 * @type {Setting}
		 */
		this.entity = entity;
		
		/**
		 * @type {StyleConfiguration}
		 */
		this.styleConfiguration = styleConfiguration;
	}
	
	/**
	 * Initialize the widget.
	 */
	initialize() {
		const $element = $('.edit-options-container #' + this.entity.getName());
		
		$element.change(() => {
			let found = false;
			
			// Update the style configuration. 
			$.each(this.styleConfiguration.getSettings(), (index, setting) => {
				$.each(setting.getEntries(), (index, entry) => {
					if (entry.getName() === this.entity.getName()) {
						// Set the default metric if necessary. 
						const defaultMetric = this.entity.getDefaultMetric();
			
						if ($element.val() !== '0' && defaultMetric !== null 
							&& $element.val().replace(/[^0-9.+ ]/g, '')	=== $element.val()) {
							$element.val($element.val() + defaultMetric);
						}
						entry.setValue($element.val());
						found = true;
						return false;
					}
				});
				
				if (found) {
					return false;
				}
			});
		});
		
		// Add setting tags to the parent ".form-group" element.
		if (this.entity.getTags() !== null) {
			const tags = this.entity.getTags().join('|');
			$element.parents('.form-group').attr('data-tags', tags);
		}
	}
	
	/**
	 * Get Widget HTML
	 *
	 * @return {String}
	 */
	getHtml() {
		const template = $('#widget-text-template').html();
		return Mustache.render(template, this.entity);
	}
}
